using System.Collections.Generic;
using System.Text;
using System;
using System.IO;
using System.Threading;
using System.Runtime.InteropServices;
using WinImageSample;

namespace WinImage
{
    public enum OverwriteType { YES = 0, NO = 1, PROMPT = 2 };
    public enum CallBackResult { OK = 1, Cancel = 2, Abort = 3, Retry = 4, Ignore = 5, Yes = 6, No = 7 };
    public class ISOImage : IDisposable
    {
        #region Variable Declarations
            private WinImageError wiError = new WinImageError();
            private IsoObjectCollection FileCol = new IsoObjectCollection(null);
            private int iFileID = 0;
            private int iFolderID = 0;
            private IntPtr iImgHandle = (IntPtr)(-1);
            public OverwriteType Overwrite = OverwriteType.YES;
            private IsoObject Root = null;
            private string sFileName = "";
            private string sFilePath = "";
            public bool Silent = false;
        #endregion
        #region Properties
            public IsoObjectCollection Files { get { return FileCol; } }
            public string ImageFileName { get { return sFileName; } }
            public string ImageFilePath { get { return sFilePath; } }
            public IsoObject ImageRoot { get { return Root; } }
            public string Label
            {
                get
                {
                    if (iImgHandle != IntPtr.Zero)
                    {
                        StringBuilder ImageLabel = new StringBuilder(' ', 255);
                        Wimadll.GetLabel(iImgHandle, ImageLabel);
                        return ImageLabel.ToString().Trim();
                    }
                    else { return ""; }
                }
                set { Wimadll.SetLabel(iImgHandle, value); }
            }
            public WinImageError LastError { get { return wiError; } }
            public int SelectedFiles { get { return Root.SelectedFiles; } }
            public int SelectedFolders { get { return Root.SelectedFolders; } }
            public long SelectedSize { get { return Root.SelectedSize; } }
            public int TotalFiles { get { return iFileID; } }
            public int TotalFolders { get { return iFolderID; } }
            public long TotalSize { get { return Root.Size; } }
        #endregion
        #region Public Methods
            public void DisplayImage(System.Windows.Forms.TreeView DisplayTree)
            {
                DisplayTree.Nodes.Clear();
                ThreeStateTreeNode dRoot = new ThreeStateTreeNode(this.ImageFileName);
                dRoot.Name = "root";
                DisplayTree.Nodes.Add(dRoot);
                DisplayImageNodes(dRoot, this.ImageRoot);
                dRoot.Expand();
            }
            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }
            public bool ExtractFiles() { return ExtractFiles(sFilePath); }
            public bool ExtractFiles(string Destination)
            {
                if (iImgHandle == IntPtr.Zero)
                {
                    wiError.ErrorCause = "No Image Loaded For File Extraction.";
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.InvalidOperation;
                    return false;
                }
                if (SelectedFiles == 0 && SelectedFolders == 0)
                {
                    wiError.ErrorCause = "No Files Or Folders Selected For Extraction.";
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.InvalidOperation;
                    return false;
                }
                if (!Path.IsPathRooted(Destination)) { Destination = Path.Combine(sFilePath, Destination); }
                if (Root.SelectedSize > WinImageHelpers.GetFreeSpace(Destination))
                {
                    wiError.ErrorCause = "Insufficient Space For Extracted Files.";
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.InvalidOperation;
                    return false;
                }
                if (!Directory.Exists(Destination)) { Directory.CreateDirectory(Destination); }
                BulkWorker objWork = new BulkWorker(iImgHandle);
                objWork.Destination = Destination;
                if (CheckExistance(Destination, Root))
                {
                    //Exisitng File(s), so decide what to do
                    switch (Overwrite)
                    {
                        case OverwriteType.PROMPT:
                            if (Silent)
                            {
                                wiError.ErrorCause = "Prompting For Overwrite Is Not Allowed When The 'Silent' Property Is Set To True.";
                                wiError.ErrorSource = "ExtractFiles";
                                wiError.ErrorType = WinImageError.EType.InvalidOperation;
                                return false;
                            }
                            string sPrompt = "One Or More Of The Files Selected For Extraction Already Exist In The Destination Directory.\nAre You Sure You Wish To Overwrite These Files?";
                            if (System.Windows.Forms.MessageBox.Show(sPrompt, "Overwrite Files", System.Windows.Forms.MessageBoxButtons.YesNo) == System.Windows.Forms.DialogResult.No)
                            {
                                //IF they don't want to overwrite then kill the display and exit with an error for information back to the calling procedure
                                wiError.ErrorCause = "One Or More Files Already Exist In The Destination Directory.";
                                wiError.ErrorSource = "ExtractFiles";
                                wiError.ErrorType = WinImageError.EType.InvalidOperation;
                                return false;
                            }
                            //Otherwise, temporarily override the user's OverWrite value and tell child procedures to always overwrite (since that is what the user requested if we got here).
                            break;
                        case OverwriteType.YES:
                            break;
                        default:
                            wiError.ErrorCause = "One Or More Files Already Exist In The Destination Directory.";
                            wiError.ErrorSource = "ExtractFiles";
                            wiError.ErrorType = WinImageError.EType.InvalidOperation;
                            return false;
                    }
                }
                objWork.Root = Root;
                try
                {
                    if (!Silent)
                    {
                        ProgressDialog.TwoStageProgressWindow pDisplay = new ProgressDialog.TwoStageProgressWindow();
                        objWork.ProgDisplay = (pDisplay as ProgressDialog.IProgressCallback);
                        Thread Work = new Thread(new ThreadStart(objWork.ExtractFiles));
                        Work.Start();
                        pDisplay.ShowDialog();
                        pDisplay = null;
                        Work = null;
                    }
                    else { objWork.ExtractFiles(); }
                }
                catch (InvalidDataException e)
                {
                    wiError.ErrorCause = e.Message;
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.InvalidData;
                    return false;
                }
                catch (InvalidOperationException e)
                {
                    wiError.ErrorCause = e.Message;
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.InvalidOperation;
                    return false;
                }
                catch (Exception e)
                {
                    wiError.ErrorCause = e.Message;
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = WinImageError.EType.Other;
                    return false;
                }
                bool bResult = objWork.Success;
                if (!bResult)
                {
                    wiError.ErrorCause = objWork.LastError.ErrorCause;
                    wiError.ErrorSource = "ExtractFiles";
                    wiError.ErrorType = objWork.LastError.ErrorType;
                }
                objWork = null;
                return bResult;
            }
            ~ISOImage() { Dispose(); }
            public ISOImage(string ImageName)
            {
                if (!File.Exists(ImageName))
                {
                    wiError.ErrorCause = "Specified Image Does Not Exist Or Is Not Accessible.\n" + ImageName;
                    wiError.ErrorSource = "Load ISOImage";
                    wiError.ErrorType = WinImageError.EType.InvalidData;
                    return;
                }
                sFileName = System.IO.Path.GetFileName(ImageName);
                sFilePath = System.IO.Path.GetDirectoryName(ImageName);
                iImgHandle = Wimadll.CreateCDIsoIma(ImageName);
                StringBuilder rootDir = new StringBuilder(' ', Wimadll.MAXLFN);
                if (Wimadll.GetCurDir(iImgHandle, rootDir, (uint)Wimadll.MAXLFN))
                {
                    IsoObject root = new IsoObject();
                    root.ShortName = rootDir.ToString();
                    root.LongName = rootDir.ToString();
                    root.ObjectType = IsoObject.ObjType.otDir;
                    root.Key = "root";
                    FileCol.Add(root.Key, root);
                    Root = FileCol["root"];
                    ReadFileList(Root, "root");
                }
            }
        #endregion
        #region Private Methods
            private bool CheckExistance(string DestinationPath, IsoObject isoCurRoot)
            {
                bool bExist = false;
                foreach (IsoObject Item in isoCurRoot.Children)
                {
                    string strOutput = Path.Combine(DestinationPath,Item.LongName);
                    if (Item.ObjectType == IsoObject.ObjType.otDir) { bExist = bExist || CheckExistance(strOutput, Item); } else { bExist = bExist || File.Exists(strOutput); }
                    if (bExist) { break; }
                }
                return bExist;
            }
            private void DisplayImageNodes(ThreeStateTreeNode DisplayRoot, IsoObject ImageRoot)
            {
                foreach (IsoObject ImageChild in ImageRoot.Children)
                {
                    ThreeStateTreeNode curNode = new ThreeStateTreeNode(ImageChild.LongName);
                    curNode.Name = ImageChild.Key;
                    DisplayRoot.Nodes.Add(curNode);
                    if (ImageChild.ObjectType == IsoObject.ObjType.otDir) { DisplayImageNodes(curNode, ImageChild); }
                }
            }
            protected virtual void Dispose(bool disposing)
            {
                if (iImgHandle != IntPtr.Zero)
                {
                    Wimadll.DeleteIma(iImgHandle);
                    iImgHandle = (IntPtr)(-1);
                }
                if (FileCol.Count > 0) { FileCol.Clear(); }
                iFileID = 0;
                iFolderID = 0;
                Root = null;
            }
            private void ReadFileList(IsoObject isoCurRoot, string sParentID)
            {
                long lFiles = Wimadll.GetNbEntryCurDir(iImgHandle);
                Wimadll.DIRINFO[] lpDi = new Wimadll.DIRINFO[lFiles];
                if (Wimadll.GetDirInfo(iImgHandle, lpDi, Wimadll.SORT_NAME))
                {
                    for (int i = 0; i < lFiles; i++)
                    {
                        IsoObject curItem = new IsoObject();
                        curItem.LongName = new String(lpDi[i].longname).Trim('\0');
                        curItem.ShortName = new String(lpDi[i].szCompactName).Trim('\0');
                        if ((curItem.ShortName != "..") && (curItem.ShortName != "."))
                        {
                            curItem.Position = lpDi[i].uiPosInDir;
                            if (!lpDi[i].fIsSubDir)
                            {
                                curItem.ObjectType = IsoObject.ObjType.otFile;
                                curItem.Key = sParentID + ":F" + iFileID.ToString();
                                curItem.Size = (long)lpDi[i].dwSize;
                                iFileID++;
                                //Console.WriteLine(curItem.Key + "-" + curItem.LongName + "-" + curItem.Size.ToString());
                            }
                            else
                            {
                                curItem.ObjectType = IsoObject.ObjType.otDir;
                                curItem.Key = sParentID + ":D" + iFolderID.ToString();
                                iFolderID++;
                                Wimadll.ChszDir(iImgHandle, curItem.ShortName);
                                ReadFileList(curItem, curItem.Key);
                                Wimadll.ChDir(iImgHandle, Wimadll.CDM_UPPER);
                            }
                            isoCurRoot.Children.Add(curItem.Key,curItem);
                        }
                    }
                }
            }
        #endregion
    }

    public class IsoObject
    {
        #region Declarations
            public enum ObjType { otDir = 0, otFile = 1 };
            private bool bSelected = false;
            private IsoObjectCollection iChildren;
            private ObjType iObjectType = ObjType.otDir;
            private IsoObject ioParent = null;
            private uint iPosition = 0;
            private int iSelFiles = 0;
            private int iSelFolders = 0;
            private long lSelSize = 0;
            private long lSize = 0;
            private string sKey = "";
            private string sLongName = "";
            private string sShortName = "";
        #endregion
        #region Object Constructor/Destructor
		    internal IsoObject() { iChildren = new IsoObjectCollection(this); }
            ~IsoObject()
            {
                if (iChildren != null)
                {
                    iChildren.Clear();
                    iChildren = null;
                }
            }
 	    #endregion
        #region Properties
            public IsoObjectCollection Children
            {
                get { if (iObjectType == ObjType.otDir) { return iChildren; } else { return null; } }
            }
            public int FileCount { get { return (iChildren != null ? iChildren.iFiles : 0); } }
            public int FolderCount { get { return (iChildren != null ? iChildren.iFolders : 0); } }
            public string Key
            {
                get { return sKey; }
                internal set { sKey = value; }
            }
            public string LongName
            {
                get { return sLongName; }
                set
                {
                    sLongName = value;
                    if (sLongName.IndexOf('\0') > 0) { sLongName = sLongName.Substring(0, sLongName.IndexOf('\0')); }
                }
            }
            public ObjType ObjectType
            {
                get { return iObjectType; }
                internal set
                {
                    iObjectType = value;
                    if (iObjectType == ObjType.otFile)
                    {
                        iChildren.Clear();
                        iChildren = null;
                        iSelFiles = 0;
                        iSelFolders = 0;
                    }
                    else
                    {
                        if (iChildren == null) { iChildren = new IsoObjectCollection(this); }
                    }
                }
            }
            public IsoObject Parent
            {
                get { return ioParent; }
                internal set { ioParent = value; }
            }
            public uint Position
            {
                get { return iPosition; }
                internal set { iPosition = value; }
            }
            public bool Selected
            {
                get { return bSelected; }
                set
                {
                    if (iObjectType == ObjType.otDir) { foreach (IsoObject Item in iChildren) { Item.Selected = value; } }
                    if (bSelected != value)
                    {
                        bSelected = value;
                        if (ioParent != null)
                        {
                            int Operator = value ? 1 : -1;
                            if (iObjectType == ObjType.otDir) { ioParent.SelectedFolders += Operator; }
                            else
                            {
                                ioParent.SelectedFiles += Operator;
                                ioParent.SelectedSize += (Operator * lSize);
                            }
                        }
                    }
                }
            }
            internal bool SelectedChildren { get { return (iSelFolders + iSelFiles) > 0; } }
            internal int SelectedFiles
            {
                get { return iSelFiles; }
                set
                {
                    int diff = value - iSelFiles;
                    iSelFiles = value;
                    if (ioParent != null) { ioParent.SelectedFiles += diff; }
                }
            }
            internal int SelectedFolders
            {
                get { return iSelFolders; }
                set
                {
                    int diff = value - iSelFolders;
                    iSelFolders = value;
                    if (ioParent != null) { ioParent.SelectedFolders += diff; }
                }
            }
            internal long SelectedSize
            {
                get { return lSelSize; }
                set
                {
                    long Diff = value - lSelSize;
                    lSelSize = value;
                    if (ioParent != null) { ioParent.SelectedSize += Diff; }
                }
            }
            public string ShortName
            {
                get { return sShortName; }
                set
                {
                    sShortName = value;
                    if (sShortName.IndexOf('\0') > 0) { sShortName = sShortName.Substring(0, sShortName.IndexOf('\0')); }
                }
            }
            public long Size
            {
                get { return lSize; }
                internal set
                {
                    long diff = value - lSize;
                    lSize = value;
                    if (ioParent != null) { ioParent.Size += diff; }
                }
            }
        #endregion
        #region Class OverRides
            public override bool Equals(object obj)
            {
                if (obj.GetType().ToString() != "WinImage.IsoObject") { return false; } else { return ((obj as IsoObject).Key == Key); }
            }
            public override int GetHashCode() { return base.GetHashCode(); }
	    #endregion    
    }
    public class IsoObjectCollection:Dictionary<string, IsoObject>
    {
        internal int iFiles = 0;
        internal int iFolders = 0;
        private IsoObject Parent = null;
        internal IsoObjectCollection(IsoObject ParentObject)
        {
            Parent = ParentObject;
        }
        public new void Add(string Key, IsoObject Item)
        {
            Item.Parent = this.Parent;
            if (Item.ObjectType == IsoObject.ObjType.otDir) { iFolders++; } else { iFiles++; }
            if (Item.Parent != null) { Item.Parent.Size += Item.Size; }
            base.Add(Key, Item);
        }
        public new void Clear()
        {
            foreach (IsoObject item in base.Values)
            {
                if ((item.ObjectType == IsoObject.ObjType.otDir) && (item.Children!=null)) { item.Children.Clear(); }
            }
            base.Clear();
        }
        public new bool ContainsKey(string Key)
        {
            string[] IDs = Key.Split(':');
            string lKey = (Parent != null ? Parent.Key + ":" : "") + IDs[0];
            if (base.ContainsKey(lKey))
            {
                string cKey = "";
                for (int i = 1; i <= IDs.GetUpperBound(0); i++) { cKey += IDs[i] + (i < IDs.GetUpperBound(0) ? ":" : ""); }
                if (cKey.Length == 0) { return true; } else { return base[lKey].Children.ContainsKey(cKey); }
            }
            else { return false; }
        }
        public new bool ContainsValue(IsoObject Item) { return this.ContainsValue(Item.LongName); }
        public bool ContainsValue(string Item) { return (this.LocateValue(Item) != null); }
        public new IEnumerator<IsoObject> GetEnumerator()
        {
            foreach (IsoObject item in base.Values) { yield return item; }
        }
        public IsoObject LocateValue(string ItemName)
        {
            ItemName.Replace('/', '\\');
            string[] Folders = ItemName.Split('\\');
            string bPath = Folders[0];
            string cPath = "";
            for (int i = 1; i <= Folders.GetUpperBound(0); i++) { cPath += Folders[i] + (i < Folders.GetUpperBound(0) ? "\\" : ""); }
            foreach (IsoObject Item in base.Values)
            {
                if (Item.LongName.ToLower() == bPath.ToLower())
                {
                    if (cPath == String.Empty) { return Item; } else { return Item.Children.LocateValue(cPath); }
                }
            }
            return null;
        }
        public new void Remove(string Key)
        {
            string[] IDs = Key.Split(':');
            string lKey = (Parent != null ? Parent.Key + ":" : "") + IDs[0];
            if (base.ContainsKey(lKey))
            {
                string cKey = "";
                for (int i = 1; i <= IDs.GetUpperBound(0); i++) { cKey += IDs[i] + (i < IDs.GetUpperBound(0) ? ":" : ""); }
                if (cKey.Length == 0) { base.Remove(lKey); } else { base[lKey].Children.Remove(cKey); }
            }
        }
        public new IsoObject this[string Key]
        {
            get
            {
                string[] IDs = Key.Split(':');
                string lKey = (Parent != null ? Parent.Key + ":" : "") + IDs[0];
                if (base.ContainsKey(lKey))
                {
                    string cKey = "";
                    for (int i = 1; i <= IDs.GetUpperBound(0); i++) { cKey += IDs[i] + (i < IDs.GetUpperBound(0) ? ":" : ""); }
                    if (cKey.Length == 0) { return base[lKey]; } else { return base[lKey].Children[cKey]; }
                }
                else { return null; }
            }
        }
    }

    internal class BulkWorker
    {
        #region Declarations
            private string Action = "";
            private Wimadll.WimCB CallBack;
            public string Destination = "";
            private int iCBResult = (int)WinImage.CallBackResult.OK;
            private IntPtr iImgHandle = IntPtr.Zero;
            private int iProcCount = 0;
            public ProgressDialog.IProgressCallback ProgDisplay = null;
            public IsoObject Root = null;
            public bool Success = false;
            public WinImageError LastError = new WinImageError();
            private const int DWEV_NEXTFILE = 0x7FFF0010;
        #endregion
        public BulkWorker(IntPtr Handle)
        {
            iImgHandle = Handle;
            CallBack = new Wimadll.WimCB(Progress);
        }
        public void ExtractFiles()
        {
            if (ProgDisplay != null)
            {
                ProgDisplay.Begin(true);
                ProgDisplay.SetRange_Primary(0, Root.SelectedFiles + Root.SelectedFolders);
                ProgDisplay.SetRange_Secondary(0, 100);
                ProgDisplay.SetCaption("Extracting Files");
            }
            Action = "Extracting: ";
            try
            {
                Success = ExtFiles(Destination, Root);
            }
            catch (InvalidDataException e)
            {
                LastError.ErrorCause = e.Message;
                LastError.ErrorSource = "ExtractFiles";
                LastError.ErrorType = WinImageError.EType.InvalidData;
                Success= false;
            }
            catch (InvalidOperationException e)
            {
                LastError.ErrorCause = e.Message;
                LastError.ErrorSource = "ExtractFiles";
                LastError.ErrorType = WinImageError.EType.InvalidOperation;
                Success = false;
            }
            catch (Exception e)
            {
                LastError.ErrorCause = e.Message;
                LastError.ErrorSource = "ExtractFiles";
                LastError.ErrorType = WinImageError.EType.Other;
                Success = false;
            }
            if (ProgDisplay != null) { ProgDisplay.End(); }
        }
        private bool ExtFiles(string sDestPath, IsoObject isoCurParent)
        {
            bool Success = true;
            foreach (IsoObject item in isoCurParent.Children)
            {
                if (iCBResult == (int)WinImage.CallBackResult.Cancel) { Success = false; break; }
                if (item.Selected || item.SelectedChildren)
                {
                    if (item.ObjectType == IsoObject.ObjType.otDir)
                    {
                        string sOutputDir = Path.Combine(sDestPath, item.LongName);
                        if (!Directory.Exists(sOutputDir)) { Directory.CreateDirectory(sOutputDir); }
                        if (item.Selected) { iCBResult = CallBack(DWEV_NEXTFILE, (uint)++iProcCount, 0, 0, 0); }
                        if (iCBResult == (int)WinImage.CallBackResult.Cancel) { Success = false; break; }
                        if (item.SelectedChildren)
                        {
                            Wimadll.ChszDir(iImgHandle, item.ShortName);
                            Success = Success && ExtFiles(sOutputDir, item);
                            Wimadll.ChDir(iImgHandle, Wimadll.CDM_UPPER);
                        }
                    }
                    else
                    {
                        string sOutputFile = Path.Combine(sDestPath, item.LongName);
                        StringBuilder OFile = new StringBuilder(260);
                        if (File.Exists(sOutputFile)){File.Delete(sOutputFile);}
                        Success = Success && Wimadll.ExtractFileCB(iImgHandle, CallBack, IntPtr.Zero, item.Position, sDestPath, OFile);
                        iCBResult = CallBack(DWEV_NEXTFILE, (uint)++iProcCount, 0, 0, 0);
                        if (iCBResult == (int)WinImage.CallBackResult.Cancel) { Success = false; break; }
                    }
                }
            }
            if (iCBResult == (int)WinImage.CallBackResult.Cancel) { throw new Exception("Operation Aborted By The User."); }
            return Success;
        }
        private int Progress(uint dwEvent, uint dwEventParam, uint dwWin32Err, int lpParam, int lpUserParam)
        {
            switch (dwEvent)
            {
                case DWEV_NEXTFILE:
                    if (ProgDisplay != null) { ProgDisplay.Increment_Primary(1); }
                    break;
                case Wimadll.DWEV_PROGRESSPERCENT:
                    if (ProgDisplay != null)
                    {
                        if (ProgDisplay.Stages == 2) { ProgDisplay.StepTo_Secondary((int)dwEventParam); } else { ProgDisplay.StepTo_Primary((int)dwEventParam); }
                        Wimadll.PROGRESSFILE_SUPINFO psInfo = (Wimadll.PROGRESSFILE_SUPINFO)Marshal.PtrToStructure((IntPtr)lpParam, typeof(Wimadll.PROGRESSFILE_SUPINFO));
                        if (psInfo.dwCurrentPos == 0) { ProgDisplay.SetText(Action + psInfo.lpszFullName + "\nTo: " + psInfo.lpszName); }
                    }
                    break;
                default:
                    Console.WriteLine("Callback Event: " + dwEvent.ToString());
                    break;
            }
            if (ProgDisplay == null) { return (int)WinImage.CallBackResult.OK; } else { return ProgDisplay.IsAborting ? (int)WinImage.CallBackResult.Cancel : (int)WinImage.CallBackResult.OK; }
        }
    }
    /// <summary>
    /// This class is used to return information about
    /// the last error that was encountered by the WinImage
    /// Library.
    /// </summary>
    public class WinImageError
    {
        /// <summary>
        /// This type defines the kinds of errors that
        /// can be returned by the PDTB Library.
        /// </summary>
        public enum EType
        {
            /// <summary>This means that no error has occured.</summary>
            None = 0,
            /// <summary>This means that some of the data entered by the calling application is either incomplete or invalid.</summary>
            InvalidData = 1,
            /// <summary>This means that an operation was performed illegally.  Perhaps an attempt to encrypt was made before the Book Key was set, or to Randomize Names with a Key that doesn't support that security level.</summary>
            InvalidOperation = 2,
            /// <summary>This means that an unexpected exception was encountered.</summary>
            Other = 3
        }
        private EType ErrType;
        private string ErrCause;
        private string ErrSource;
        /// <summary>
        /// This Read-Only property allows applications to retrieve
        /// information about the cause of the last error.
        /// </summary>
        /// <value>The cause of the last error.</value>
        public string ErrorCause
        {
            get { return ErrCause; }
            internal set { ErrCause = value; }
        }
        /// <summary>
        /// This Read-Only property allows applications to retrieve
        /// information about the source of the last error.
        /// </summary>
        /// <value>The source of the last error.</value>
        public string ErrorSource
        {
            get { return ErrSource; }
            internal set { ErrSource = value; }
        }
        /// <summary>
        /// This Read-Only property allows applications to retrieve
        /// information about the type of the last error.
        /// </summary>
        /// <remarks>
        /// See the description of the <see cref="EType"/> enumerated
        /// type for more information about the possible error types.
        /// </remarks>
        /// <value>The type of the last error.</value>
        public EType ErrorType
        {
            get { return ErrType; }
            internal set { ErrType = value; }
        }
        /// <summary>
        /// Initializes a new instance of the <see cref="PDTBError"/> class.
        /// </summary>
        internal WinImageError()
        {
            //Making this constructor an internal method allows outside
            //applications to see this class, since it is a public class,
            //but they can not CREATE an instance of this class, only 
            //reference instances created by the library.
            ErrCause = "";
            ErrSource = "";
            ErrType = EType.None;
        }
    }

    internal static class WinImageHelpers
    {
        [DllImport("KERNEL32.DLL", EntryPoint = "GetDiskFreeSpaceExW", CharSet = CharSet.Unicode)]
        private static extern bool GetDiskFreeSpace(string drive, out long FreeUserBytes, out long TotalBytes, out long TotalFreeBytes);
        internal static long GetFreeSpace(string sDestination)
        {
            long lUserBytes = 0;
            long lTotalBytes = 0;
            long lFreeBytes = 0;
            bool bRet = GetDiskFreeSpace(Path.GetPathRoot(sDestination), out lUserBytes, out lTotalBytes, out lFreeBytes);
            //If we get a free space value back from the GetDiskFreeSpace function, then
            //act accordingly, otherwise, ALWAYS return 0.
            if (bRet) { return lUserBytes; } else { return 0; }
        }
    }
}


