using System;
using System.Windows.Forms;
using System.Drawing;
using System.Runtime.InteropServices;
using System.ComponentModel;

namespace WinImageSample
{
    [FlagsAttribute]
    public enum CheckBoxState
    {
        Unchecked = 1,
        Checked = 2,
        Indeterminate = CheckBoxState.Unchecked | CheckBoxState.Checked
    }

    /// <summary>
    /// ThreeStateTreeNode inherits from <see cref="http://msdn2.microsoft.com/en-us/library/sc9ba94b(vs.80).aspx">TreeView</see>
    /// and adds the ability to cascade state changes to related nodes, i.e. child nodes and or parent nodes, as well as to optionally
    /// use the three state functionality.
    /// </summary>
    public class ThreeStateTreeView : TreeView
    {
        ImageList _imageList;
        private bool bHandleCreated = false;
        #region Constructors
        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeView class in addition to intializing
        /// the base class (<see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treeview.treeview(VS.80).aspx">TreeView Constructor</see>). 
        /// </summary>
        public ThreeStateTreeView()
            : base()
        {
            InitializeComponent();
            base.CheckBoxes = true;
            _imageList = new ImageList();
            _imageList.ImageSize = new Size(16, 16);
            if (!NativeMethods.VisualStylesEnabled()) { _imageList = ilChecks; }
            else
            {
                _imageList.Images.Add(ilChecks.Images["Checked0"]);
                UXTheme.LoadCheckBoxImage(_imageList, UXTheme.CheckBoxState.UncheckedNormal);
                UXTheme.LoadCheckBoxImage(_imageList, UXTheme.CheckBoxState.CheckedNormal);
                UXTheme.LoadCheckBoxImage(_imageList, UXTheme.CheckBoxState.MixedNormal);
            }
        }
        #endregion

        private ImageList ilChecks;
        private IContainer components;



        #region Public Properties
        /// <summary>
        /// Flag. If true, use three state checkboxes, otherwise, use the default behavior of the TreeView and associated TreeNodes.
        /// </summary>
        private bool mUseThreeStateCheckBoxes = true;
        [Category("Three State TreeView"),
        Description("Flag. If true, use three state checkboxes, otherwise, use the default behavior of the TreeView and associated TreeNodes."),
        DefaultValue(true),
        TypeConverter(typeof(bool)),
        Editor("System.Boolean", typeof(bool))]
        public bool UseThreeStateCheckBoxes
        {
            get { return this.mUseThreeStateCheckBoxes; }
            set { this.mUseThreeStateCheckBoxes = value; }
        }
        #endregion

        #region Overrides
        public new ImageList ImageList
        {
            get { return base.ImageList; }
            set
            {
                base.ImageList = value;
                this.SetCheckImageList();
            }
        }
        /// <summary>
        /// Raises the AfterCheck event.
        /// </summary>
        /// <param name="e">A <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treevieweventargs.aspx">TreeViewEventArgs</see> containing the event data.</param>
        protected override void OnAfterCheck(TreeViewEventArgs e)
        {

            if (this.UseThreeStateCheckBoxes)
            {
                switch (e.Action)
                {
                    case TreeViewAction.ByKeyboard:
                    case TreeViewAction.ByMouse:
                        {
                            if (e.Node is ThreeStateTreeNode)
                            {
                                // Toggle to the next state.
                                ThreeStateTreeNode tn = e.Node as ThreeStateTreeNode;
                                tn.Toggle();
                            }

                            break;
                        }
                    case TreeViewAction.Collapse:
                    case TreeViewAction.Expand:
                    case TreeViewAction.Unknown:
                    default:
                        {
                            // Do nothing.
                            break;
                        }
                }
            }
            base.OnAfterCheck(e);
        }
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);
            bHandleCreated = true;
            SetCheckImageList();
        }
        #endregion
        #region Hide no longer appropriate properties from Designer
        [Browsable(false)]
        public new bool CheckBoxes
        {
            get { return base.CheckBoxes; }
            internal set { base.CheckBoxes = value; }
        }
        [Browsable(false)]
        public new int ImageIndex
        {
            get { return base.ImageIndex; }
            set { base.ImageIndex = value; }
        }
        [Browsable(false)]
        public new int SelectedImageIndex
        {
            get { return base.SelectedImageIndex; }
            set { base.SelectedImageIndex = value; }
        }
        [Browsable(false)]
        public new ImageList StateImageList
        {
            get { return base.StateImageList; }
            internal set { base.StateImageList = value; }
        }
        #endregion

        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(ThreeStateTreeView));
            this.ilChecks = new System.Windows.Forms.ImageList(this.components);
            this.SuspendLayout();
            // 
            // ilChecks
            // 
            this.ilChecks.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("ilChecks.ImageStream")));
            this.ilChecks.TransparentColor = System.Drawing.Color.Transparent;
            this.ilChecks.Images.SetKeyName(0, "Checked0");
            this.ilChecks.Images.SetKeyName(1, "Checked1");
            this.ilChecks.Images.SetKeyName(2, "Checked2");
            this.ilChecks.Images.SetKeyName(3, "Checked3");
            this.ResumeLayout(false);

        }
        private void SetCheckImageList()
        {
            if (!bHandleCreated) return;
            NativeMethods.SendMessage(
                    new HandleRef(this, this.Handle),
                    (int)NativeMethods.Messages.TVM_SETIMAGELIST,
                    (int)NativeMethods.TreeViewImageListType.TVSIL_STATE,
                    (int)_imageList.Handle);
        }
    }

    /// <summary>
    /// ThreeStateTreeNode inherits from <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treenode.aspx">TreeNode</see>
    /// and adds the ability to support a third, indeterminate state as well as optionally cascading state changes to related nodes, i.e.
    /// child nodes and or parent nodes, as determined by this instance's related parent TreeView settings, CascadeNodeChecksToChildNodes and
    /// CascadeNodeChecksToParentNode.
    /// </summary>
    public class ThreeStateTreeNode : TreeNode
    {
        #region Constructors
        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeNode class in addition to intializing
        /// the base class (<see cref="http://msdn2.microsoft.com/en-us/library/bk8h64c9.aspx">TreeNode Constructor</see>). 
        /// </summary>
        public ThreeStateTreeNode()
            : base()
        {
            this.CommonConstructor();
        }

        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeNode class with a string for the text label to display in addition to intializing
        /// the base class (<see cref="http://msdn2.microsoft.com/en-us/library/ytx906df.aspx">TreeNode Constructor</see>). 
        /// </summary>
        /// <param name="text">The string for the label of the new tree node.</param>
        public ThreeStateTreeNode(string text)
            : base(text)
        {
            this.CommonConstructor();
        }

        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeNode class with a string for the text label to display 
        /// and an array of child ThreeStateTreeNodes in addition to intializing the base class 
        /// (<see cref="http://msdn2.microsoft.com/en-us/library/774ty506.aspx">TreeNode Constructor</see>). 
        /// </summary>
        /// <param name="text">The string for the label of the new tree node.</param>
        /// <param name="children">An array of child ThreeStateTreeNodes.</param>
        public ThreeStateTreeNode(string text, ThreeStateTreeNode[] children)
            : base(text, children)
        {
            this.CommonConstructor();
        }

        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeNode class with a string for the text label to display 
        /// and the selected and unselected image indexes in addition to intializing the base class 
        /// (<see cref="http://msdn2.microsoft.com/en-us/library/8dfy3k5t.aspx">TreeNode Constructor</see>). 
        /// </summary>
        /// <param name="text">The string for the label of the new tree node.</param>
        /// <param name="imageIndex">The image index of the unselected image in the parent TreeView's <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treeview.imagelist.aspx">ImageList</see>.</param>
        /// <param name="selectedImageIndex">The image index of the selected image in the parent TreeView's <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treeview.imagelist.aspx">ImageList</see>.</param>
        public ThreeStateTreeNode(string text, int imageIndex, int selectedImageIndex)
            : base(text, imageIndex, selectedImageIndex)
        {
            this.CommonConstructor();
        }

        /// <summary>
        /// Initializes a new instance of the ThreeStateTreeNode class with a string for the text label to display ,
        /// the selected and unselected image indexes, and an array of child ThreeStateTreeNodes in addition to intializing the base class 
        /// (<see cref="http://msdn2.microsoft.com/en-us/library/8dfy3k5t.aspx">TreeNode Constructor</see>). 
        /// </summary>
        /// <param name="text">The string for the label of the new tree node.</param>
        /// <param name="imageIndex">The image index of the unselected image in the parent TreeView's <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treeview.imagelist.aspx">ImageList</see>.</param>
        /// <param name="selectedImageIndex">The image index of the selected image in the parent TreeView's <see cref="http://msdn2.microsoft.com/en-us/library/system.windows.forms.treeview.imagelist.aspx">ImageList</see>.</param>
        /// <param name="children">An array of child ThreeStateTreeNodes.</param>
        public ThreeStateTreeNode(string text, int imageIndex, int selectedImageIndex, ThreeStateTreeNode[] children)
            : base(text, imageIndex, selectedImageIndex, children)
        {
            this.CommonConstructor();
        }
        #endregion

        #region Initialization
        /// <summary>
        /// Performs common initialization to all constructors.
        /// </summary>
        private void CommonConstructor()
        {
        }
        #endregion

        #region Properties
        /// <summary>
        /// The current state of the checkbox.
        /// </summary>
        private CheckBoxState mState = CheckBoxState.Unchecked;
        [Category("Three State TreeView"),
        Description("The current state of the node's checkbox, Unchecked, Checked, or Indeterminate"),
        DefaultValue(CheckBoxState.Unchecked),
        TypeConverter(typeof(CheckBoxState)),
        Editor("Ascentium.Research.Windows.Components.CheckBoxState", typeof(CheckBoxState))]
        public CheckBoxState State
        {
            get { return this.mState; }
            set
            {
                if (this.mState != value)
                {
                    this.mState = value;

                    // Ensure if checkboxes are used to make the checkbox checked or unchecked.
                    // When go to a fully drawn control, this will be managed in the drawing code.
                    // Setting the Checked property in code will cause the OnAfterCheck to be called
                    // and the action will be 'Unknown'; do not handle that case.
                    //if ((this.TreeView != null) && (this.TreeView.CheckBoxes))
                    //    this.Checked = (this.mState == CheckBoxState.Checked);

                    SetState(value);
                }
            }
        }

        /// <summary>
        /// Returns the 'combined' state for all siblings of a node.
        /// </summary>
        private CheckBoxState SiblingsState
        {
            get
            {
                // If parent is null, cannot have any siblings or if the parent
                // has only one child (i.e. this node) then return the state of this 
                // instance as the state.
                if ((this.Parent == null) || (this.Parent.Nodes.Count == 1))
                    return this.State;

                // The parent has more than one child.  Walk through parent's child
                // nodes to determine the state of all this node's siblings,
                // including this node.
                CheckBoxState state = 0;
                foreach (TreeNode node in this.Parent.Nodes)
                {
                    ThreeStateTreeNode child = node as ThreeStateTreeNode;
                    if (child != null)
                        state |= child.State;

                    // If the state is now indeterminate then know there
                    // is a combination of checked and unchecked nodes
                    // and no longer need to continue evaluating the rest
                    // of the sibling nodes.
                    if (state == CheckBoxState.Indeterminate)
                        break;
                }

                return (state == 0) ? CheckBoxState.Unchecked : state;
            }
        }

        #endregion

        #region Methods
        /// <summary>
        /// Manages state changes from one state to the next.
        /// </summary>
        /// <param name="fromState">The state upon which to base the state change.</param>
        public void Toggle(CheckBoxState fromState)
        {
            switch (fromState)
            {
                case CheckBoxState.Indeterminate:
                case CheckBoxState.Unchecked:
                    {
                        this.State = CheckBoxState.Checked;
                        break;
                    }
                case CheckBoxState.Checked:
                default:
                    {
                        this.State = CheckBoxState.Unchecked;
                        break;
                    }
            }

            this.UpdateStateOfRelatedNodes();
        }

        /// <summary>
        /// Manages state changes from one state to the next.
        /// </summary>
        public new void Toggle() { this.Toggle(this.State); }

        /// <summary>
        /// Manages updating related child and parent nodes of this instance.
        /// </summary>
        public void UpdateStateOfRelatedNodes()
        {
            ThreeStateTreeView tv = this.TreeView as ThreeStateTreeView;
            if ((tv != null) && tv.CheckBoxes && tv.UseThreeStateCheckBoxes)
            {
                tv.BeginUpdate();

                // If want to cascade checkbox state changes to child nodes of this node and
                // if the current state is not intermediate, update the state of child nodes.
                if (this.State != CheckBoxState.Indeterminate)
                    this.UpdateChildNodeState();

                this.UpdateParentNodeState(true);

                tv.EndUpdate();
            }
        }

        /// <summary>
        /// Recursiveley update child node's state based on the state of this node.
        /// </summary>
        private void UpdateChildNodeState()
        {
            ThreeStateTreeNode child;
            foreach (TreeNode node in this.Nodes)
            {
                // It is possible node is not a ThreeStateTreeNode, so check first.
                if (node is ThreeStateTreeNode)
                {
                    child = node as ThreeStateTreeNode;
                    child.Checked = (this.State != CheckBoxState.Unchecked);
                    child.State = this.State;
                    child.UpdateChildNodeState();
                }
            }
        }

        /// <summary>
        /// Recursiveley update parent node state based on the current state of this node.
        /// </summary>
        private void UpdateParentNodeState(bool isStartingPoint)
        {
            // If isStartingPoint is false, then know this is not the initial call
            // to the recursive method as we want to force on the first time
            // this is called to set the instance's parent node state based on
            // the state of all the siblings of this node, including the state
            // of this node.  So, if not the startpoint (!isStartingPoint) and
            // the state of this instance is indeterminate (CheckBoxState.Indeterminate)
            // then know to set all subsequent parents to the indeterminate
            // state.  However, if not in an indeterminate state, then still need
            // to evaluate the state of all the siblings of this node, including the state
            // of this node before setting the state of the parent of this instance.

            ThreeStateTreeNode parent = this.Parent as ThreeStateTreeNode;
            if (parent != null)
            {
                CheckBoxState state = CheckBoxState.Unchecked;

                // Determine the new state
                if (!isStartingPoint && (this.State == CheckBoxState.Indeterminate))
                    state = CheckBoxState.Indeterminate;
                else
                    state = this.SiblingsState;

                // Update parent state if not the same.
                if (parent.State != state)
                {
                    parent.Checked = (state != CheckBoxState.Unchecked);
                    parent.State = state;
                    parent.UpdateParentNodeState(false);
                }
            }
        }
        #endregion
        private void SetState(CheckBoxState state)
        {
            //            _threeStateChecked = state;
            if (this.TreeView != null && this.TreeView.IsHandleCreated && this.Handle != IntPtr.Zero)
            {
                SetInternalState(state);
            }
        }

        private void SetInternalState(CheckBoxState state)
        {

            ThreeStateTreeView view = this.TreeView as ThreeStateTreeView;
            if (view == null)
                return;

            NativeMethods.TVITEM item = new NativeMethods.TVITEM();
            item.mask = (int)(NativeMethods.TreeViewMask.TVIF_HANDLE | NativeMethods.TreeViewMask.TVIF_STATE);
            item.hItem = this.Handle;
            item.stateMask = (int)NativeMethods.TreeViewStateMask.TVIS_STATEIMAGEMASK;
            switch (state)
            {
                case CheckBoxState.Unchecked:
                    item.state |= 0x1000;
                    break;
                case CheckBoxState.Checked:
                    item.state |= 0x2000;
                    break;
                case CheckBoxState.Indeterminate:
                    item.state |= 0x3000;
                    break;
                default:
                    throw new ArgumentOutOfRangeException("state");
            }

            NativeMethods.SendMessage(
                new HandleRef(this.TreeView, this.TreeView.Handle),
                (int)NativeMethods.Messages.TVM_SETITEM,
                0,
                ref item);

            //view.TreeViewAfterThreeStateCheckUpdate(this);
        }

    }
    internal sealed class NativeMethods
    {

        #region Enums and structs

        [Flags]
        public enum TreeViewMask
        {
            TVIF_TEXT = 0x0001,
            TVIF_IMAGE = 0x0002,
            TVIF_PARAM = 0x0004,
            TVIF_STATE = 0x0008,
            TVIF_HANDLE = 0x0010,
            TVIF_SELECTEDIMAGE = 0x0020,
            TVIF_CHILDREN = 0x0040,
            TVIF_INTEGRAL = 0x0080
        }

        [Flags]
        public enum TreeViewStateMask
        {
            TVIS_SELECTED = 0x0002,
            TVIS_CUT = 0x0004,
            TVIS_DROPHILITED = 0x0008,
            TVIS_BOLD = 0x0010,
            TVIS_EXPANDED = 0x0020,
            TVIS_EXPANDEDONCE = 0x0040,
            TVIS_EXPANDPARTIAL = 0x0080,
            TVIS_OVERLAYMASK = 0x0F00,
            TVIS_STATEIMAGEMASK = 0xF000,
            TVIS_USERMASK = 0xF000
        }

        public enum TreeViewImageListType
        {
            TVSIL_NORMAL = 0,
            TVSIL_STATE = 2
        }


        public enum Messages
        {
            TVM_SETIMAGELIST = 0x1109,
            TVM_SETITEM = 0x110d
        }


        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        public struct TVITEM
        {
            public int mask;
            public IntPtr hItem;
            public int state;
            public int stateMask;
            public IntPtr pszText;
            public int cchTextMax;
            public int iImage;
            public int iSelectedImage;
            public int cChildren;
            public IntPtr lParam;
        }
        [StructLayout(LayoutKind.Sequential)]
        public struct DLLVersionInfo
        {
            public int cbSize;
            public int dwMajorVersion;
            public int dwMinorVersion;
            public int dwBuildNumber;
            public int dwPlatformID;
        }
        #endregion

        #region Native methods


        [DllImport("user32.dll", CharSet = CharSet.Auto)]
        public static extern IntPtr SendMessage(HandleRef hWnd, int msg, int wParam, ref NativeMethods.TVITEM lParam);

        [DllImport("user32.dll", CharSet = CharSet.Auto)]
        public static extern IntPtr SendMessage(HandleRef hWnd, int msg, int wParam, int lParam);

        [DllImport("comctl32.dll")]
        public static extern IntPtr ImageList_Create(int cx, int cy, int flags, int cInitial, int cGrow);

        [DllImport("Comctl32.dll", EntryPoint = "DllGetVersion", ExactSpelling = true, CharSet = CharSet.Unicode)]
        private static extern int DllGetVersion(ref DLLVersionInfo version);


        #endregion
        public static bool VisualStylesEnabled()
        {
            OperatingSystem os = Environment.OSVersion;
            bool isAppropriateOS = os.Platform == PlatformID.Win32NT && ((os.Version.Major == 5 && os.Version.Minor >= 1) || os.Version.Major > 5);
            bool osFeatureThemesPresent = false;
            bool osThemeDLLAvailable = false;

            if (isAppropriateOS)
            {
                Version osThemeVersion = OSFeature.Feature.GetVersionPresent(OSFeature.Themes);
                osFeatureThemesPresent = osThemeVersion != null;

                DLLVersionInfo dllVersion = new DLLVersionInfo();
                dllVersion.cbSize = Marshal.SizeOf(typeof(DLLVersionInfo));
                int temp = DllGetVersion(ref dllVersion);
                osThemeDLLAvailable = dllVersion.dwMajorVersion >= 5;
            }

            return isAppropriateOS && osFeatureThemesPresent && osThemeDLLAvailable && UXTheme.IsAppThemed() && UXTheme.IsThemeActive();
        }

    }
    internal static class UXTheme
    {
        /// <summary>States available for Button.CheckBox part.summary>
        public enum CheckBoxState : int
        {
            UncheckedNormal = 1,
            UncheckedHot = 2,
            UncheckedPressed = 3,
            UncheckedDisabled = 4,
            CheckedNormal = 5,
            CheckedHot = 6,
            CheckedPressed = 7,
            CheckedDisabled = 8,
            MixedNormal = 9,
            MixedHot = 10,
            MixedPressed = 11,
            MixedDisabled = 12
        };
        /// <summary>Parts available for button control.</summary>
        public enum ButtonPart : uint
        {
            Pushbutton = 1,
            RadioButton = 2,
            Checkbox = 3,
            Groupbox = 4,
            UserButton = 5
            /*
            CommandLink = 6,
            CommandLinkGlyph = 7
            */
        };

        /// <summary>RECT structure.</summary>
        [StructLayout(LayoutKind.Sequential)]
        internal struct RECT
        {
            public int Left;
            public int Top;
            public int Right;
            public int Bottom;

            public RECT(System.Drawing.Rectangle rc)
            {
                Left = rc.X;
                Top = rc.Y;
                Right = rc.Right;
                Bottom = rc.Bottom;
            }

            public RECT(int left, int top, int right, int bottom)
            {
                this.Left = left;
                this.Top = top;
                this.Right = right;
                this.Bottom = bottom;
            }

            /// <summary>
            /// Converts the current <b>RECT</b> to a <see cref="System.Drawing.Rectangle"/>.
            /// </summary>
            public System.Drawing.Rectangle ToRectangle()
            {
                System.Drawing.Rectangle rect = System.Drawing.Rectangle.Empty;

                rect.X = Left;
                rect.Y = Top;
                rect.Width = Right - Left;
                rect.Height = Bottom - Top;

                return rect;
            }
        }

        ////Declare a private constructor so nothing can be instantiated as this class.
        //private UXTheme(){}

        /// <summary>
        /// Closes the theme data handle.
        /// </summary>
        /// <param name="hTheme">Handle to a window's specified theme data.</param>
        [DllImport("UxTheme.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern int CloseThemeData(IntPtr hTheme);
        /// <summary>
        /// Draws the background image defined by the visual style for the specified control part.
        /// </summary>
        /// <param name="hTheme">Handle to a window's specified theme data.</param>
        /// <param name="hdc">Handle to a device context (HDC) used for drawing the theme-defined background image.</param>
        /// <param name="iPartId">Value that specifies the part to draw.</param>
        /// <param name="iStateId">Value that specifies the state of the part to draw.</param>
        /// <param name="pRect">Pointer to a RECT structure that contains the rectangle, in logical coordinates, in which the background image is drawn.</param>
        /// <param name="pClipRect">Pointer to a RECT structure that contains a clipping rectangle.</param>
        [DllImport("UxTheme.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern System.Int32 DrawThemeBackground(
            IntPtr hTheme,
            IntPtr hdc,
            UInt32 iPartId,
            UInt32 iStateId,
            ref RECT pRect,
            ref RECT pClipRect
            );

        [DllImport("UxTheme.dll", ExactSpelling = true)]
        public extern static Int32 DrawThemeBackground(
            IntPtr hTheme,
            IntPtr hdc,
            UInt32 iPartId,
            UInt32 iStateId,
            ref RECT pRect,
            IntPtr pClipRect
            );

        /// <summary>
        /// Reports whether the current application's user interface displays using visual styles.
        /// </summary>
        [DllImport("UxTheme", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern bool IsAppThemed();
        /// <summary>
        /// Tests if a visual style for the current application is active.
        /// </summary>
        [DllImport("UxTheme.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern bool IsThemeActive();

        public static void LoadCheckBoxImage(System.Windows.Forms.ImageList iList, CheckBoxState state)
        {
            RECT r = new RECT(0, 0, 16, 16);
            Bitmap bmp;
            Graphics g;
            bmp = new Bitmap(16, 16);
            g = Graphics.FromImage(bmp);
            IntPtr gHandle = g.GetHdc();
            IntPtr hTheme = OpenThemeData(IntPtr.Zero, "BUTTON");
            DrawThemeBackground(hTheme, gHandle, (uint)ButtonPart.Checkbox, (uint)state, ref r, IntPtr.Zero);
            CloseThemeData(hTheme);
            g.ReleaseHdc();
            iList.Images.Add(bmp);

        }

        /// <summary>
        /// Opens the theme data for a window and its associated class.
        /// </summary>
        /// <param name="hwnd">Handle of the window for which theme data is required.</param>
        /// <param name="pszClassList">Pointer to a string that contains a semicolon-separated list of classes.</param>
        [DllImport("UxTheme.dll", CharSet = CharSet.Unicode, SetLastError = true)]
        public static extern IntPtr OpenThemeData(
            IntPtr hwnd,
            String pszClassList
            );
    }

}
